<#
.SYNOPSIS
    Tests network share accessibility for Atria services, optionally applying permissions.

.DESCRIPTION
    Retrieves distinct share paths for FSS, Citrix/HAAD, HE, or WorkSpace.
    Extracts root-level shares (\\Server\Share), tests accessibility,
    color-codes console output, and writes plain-text logs.
    When -ApplyPermissions is specified, ensures the
    '<domain>\Atria Delegated FileShare Management' group exists in local Administrators
    and applies share permissions remotely using Update-AtriaFSSSharePermissions.ps1,
    which is copied temporarily to each target server under C:\Atria\Temp.
#>

param(
    [Parameter(Mandatory = $true)]
    [ValidateSet("FSS","CitrixHAAD","HE","WorkSpace")]
    [string]$Service,

    [switch]$ApplyPermissions
)

# --- Log setup ---
if (-not $PSScriptRoot) { throw "This script must be run from a .ps1 file." }
$LogDirectory = Join-Path $PSScriptRoot "Logs"
if (-not (Test-Path $LogDirectory)) { New-Item -ItemType Directory -Force -Path $LogDirectory | Out-Null }
$timestamp = (Get-Date).ToString('yyyyMMdd_HHmmss')
$LogPath   = Join-Path $LogDirectory ("Update-AtriaServiceShares_{0}_{1}.txt" -f $Service,$timestamp)

function Write-Log {
    param(
        [Parameter(Mandatory=$true)][string]$Message,
        [ValidateSet("Default","Success","Error","Warning")][string]$Level="Default"
    )
    $ts = (Get-Date).ToString("yyyy-MM-dd HH:mm:ss")
    Add-Content -Path $LogPath -Value ("{0}`t{1}" -f $ts,$Message)
    switch ($Level) {
        "Success" { Write-Host $Message -ForegroundColor Green }
        "Error"   { Write-Host $Message -ForegroundColor Red }
        "Warning" { Write-Host $Message -ForegroundColor Yellow }
        Default   { Write-Host $Message -ForegroundColor Gray }
    }
}

function Test-IsAdminShare {
    param([string]$UNC)
    if ($UNC -match '^(\\\\[^\\]+\\[A-Z]\$)(\\.*)?$') { return $true } else { return $false }
}

# --- Service config ---
$ServiceConfig = @{
    "FSS" = @{
        "Description"   = "File Share Server paths"
        "PropertyNames" = @("FileShareServer","FSSSharePath")
    }
    "CitrixHAAD" = @{
        "Description"   = "Citrix / HAAD Home and Profile directories"
        "PropertyNames" = @(
            "homeDirectory","profileDirectory","profileHomePath","profilePath",
            "TSFileServer","TSHomeDriveTemplate","TShomePath","TSprofilePath","TSProfileTemplate"
        )
    }
    "HE" = @{
        "Description"   = "Hosted Exchange import/export root paths"
        "PropertyNames" = @("ImpExpRootPathPattern","ImpExpRootPathPattern")
    }
    "WorkSpace" = @{
        "Description"   = "WorkSpace terminal server file and profile shares"
        "PropertyNames" = @("wsTSFileServer","wsTSHomeDrivePath","wsTSProfileSharePath","wsTSProfileTemplate")
    }
}

Write-Log "=== Starting network share test for [$Service] ==="
Write-Log ("Description: " + $ServiceConfig[$Service]['Description'])
if ($ApplyPermissions) { Write-Log "Mode: ApplyPermissions = TRUE" } else { Write-Log "Mode: Dry-run (no permissions applied)" }

# --- SQL connection ---
try {
    Write-Log "Retrieving secure SQL connection string..."
    $connectionString = Get-AtriaSecret -SecretKey 'connectionString_OLM' -UseEnvironment
    $connection = New-Object System.Data.SqlClient.SqlConnection($connectionString)
    $connection.Open()
    Write-Log "Database connection established successfully." "Success"
}
catch {
    Write-Log ("ERROR: Failed to connect to database - " + $_.Exception.Message) "Error"
    exit 1
}

# --- Query distinct paths ---
$Properties = $ServiceConfig[$Service]["PropertyNames"]
$PropertyList = ($Properties -join "','")
$query = @"
SELECT DISTINCT pvf.Value
FROM propertyvalues_flat pvf
INNER JOIN Properties p ON pvf.PropertyID = p.PropertyID
WHERE p.Name IN ('$PropertyList')
  AND pvf.Value IS NOT NULL
  AND pvf.Value <> 'CortexNULL'
  AND LTRIM(RTRIM(pvf.Value)) <> ''
ORDER BY pvf.Value ASC;
"@

try {
    $command = $connection.CreateCommand()
    $command.CommandText = $query
    $adapter = New-Object System.Data.SqlClient.SqlDataAdapter $command
    $dataTable = New-Object System.Data.DataTable
    [void]$adapter.Fill($dataTable)
    $connection.Close()
    Write-Log ("Retrieved " + $dataTable.Rows.Count + " distinct values from database.") "Success"
}
catch {
    Write-Log ("ERROR executing SQL query: " + $_.Exception.Message) "Error"
    $connection.Close(); exit 1
}

if (-not $dataTable -or $dataTable.Rows.Count -eq 0) {
    Write-Log "No results returned for this service." "Warning"
    exit 0
}

# --- Build UNC share list (FSS-specific logic restored) ---
$Shares = @()

if ($Service -eq "FSS") {
    $AllValues = $dataTable | Select-Object -ExpandProperty Value -Unique
    $FileShareServers = $AllValues | Where-Object { $_ -notmatch '\\' }
    $SharePaths = $AllValues | Where-Object { $_ -match '\\' }

    foreach ($path in $SharePaths) {
        foreach ($server in $FileShareServers) {
            $resolved = $path -replace "\{FileShareServer\}", $server
            $val = $resolved.Trim()

            if ($val -match '\\\\([^\\]+)\\([A-Z])\$\\([^\\]+)') {
                $serverName = $matches[1]
                $drive = $matches[2]
                $shareName = $matches[3]
                $adminUNC = "\\$serverName\$drive`$\$shareName"
                $cleanUNC = "\\$serverName\$shareName"
                Write-Log ("Detected admin drive share: " + $adminUNC + " - also testing " + $cleanUNC) "Warning"
                $Shares += [PSCustomObject]@{ Service=$Service; Server=$serverName; UNC=$adminUNC }
                $Shares += [PSCustomObject]@{ Service=$Service; Server=$serverName; UNC=$cleanUNC }
                continue
            }

            if ($val -match '^(\\\\[^\\]+\\[^\\]+)') {
                $rootUNC = $matches[1]
                $srv = ($rootUNC -split '\\')[2]
                $Shares += [PSCustomObject]@{ Service=$Service; Server=$srv; UNC=$rootUNC }
            }
        }
    }
}
else {
    foreach ($row in $dataTable) {
        $val = $row.Value.Trim()
        if ($val -notmatch '^\\\\') { continue }

        if ($val -match '\\\\([^\\]+)\\([A-Z])\$\\([^\\]+)') {
            $serverName = $matches[1]
            $drive = $matches[2]
            $shareName = $matches[3]
            $adminUNC = "\\$serverName\$drive`$\$shareName"
            $cleanUNC = "\\$serverName\$shareName"
            Write-Log ("Detected admin drive share: " + $adminUNC + " - also testing " + $cleanUNC) "Warning"
            $Shares += [PSCustomObject]@{ Service=$Service; Server=$serverName; UNC=$adminUNC }
            $Shares += [PSCustomObject]@{ Service=$Service; Server=$serverName; UNC=$cleanUNC }
            continue
        }

        if ($val -match '^(\\\\[^\\]+\\[^\\]+)') {
            $rootUNC = $matches[1]
            $srv = ($rootUNC -split '\\')[2]
            $Shares += [PSCustomObject]@{ Service=$Service; Server=$srv; UNC=$rootUNC }
        }
    }
}

$Shares = $Shares | Sort-Object -Property UNC -Unique
Write-Log ("Prepared " + $Shares.Count + " root shares to test for [" + $Service + "].")

# --- Domain detection ---
try {
    $domain = ([System.DirectoryServices.ActiveDirectory.Domain]::GetCurrentDomain()).Name
    Write-Log ("Detected domain: " + $domain) "Success"
}
catch {
    $domain = $null
    Write-Log "Could not determine domain automatically." "Warning"
}

$principal = $null
if ($domain) { $principal = "$domain\Atria Delegated FileShare Management" }
$PermScript = "C:\Program Files\Automate101\Atria\Provisioning Engine\Scripts\Update-AtriaFSSSharePermissions.ps1"

# --- Ensure group in local Administrators ---
if ($ApplyPermissions -and $principal) {
    Write-Log ("Checking if " + $principal + " is in local Administrators...")
    try {
        $adminGroup = [ADSI]"WinNT://./Administrators,group"
        $memberExists = $false
        $netbiosDomain = ($domain -split '\.')[0]
        $groupName = "Atria Delegated FileShare Management"
        foreach ($member in @($adminGroup.psbase.Invoke("Members"))) {
            $memberName = $member.GetType().InvokeMember("Name",'GetProperty',$null,$member,$null)
            if ($memberName -eq $groupName) { $memberExists = $true; break }
        }
        if (-not $memberExists) {
            $memberPath = "WinNT://$netbiosDomain/$groupName,group"
            Write-Log ("Adding " + $principal + " to Administrators using path " + $memberPath + " ...")
            try {
                $adminGroup.Add($memberPath)
                Write-Log ($principal + " added to Administrators.") "Success"
            }
            catch {
                Write-Log ("ERROR adding ${principal}: " + $_.Exception.Message) "Error"
            }
        }
        else {
            Write-Log ($principal + " already in Administrators.") "Success"
        }
    }
    catch {
        Write-Log ("ERROR verifying Administrators group: " + $_.Exception.Message) "Error"
    }
}

# --- Test shares and apply permissions ---
$ok=0;$limited=0;$fail=0;$err=0
foreach ($item in $Shares) {
    $unc = $item.UNC
    $srv = $item.Server
    Write-Host "==============================" -ForegroundColor Cyan
    Add-Content -Path $LogPath -Value "=============================="
    Write-Log ("Testing " + $unc + "...")

    try {
        if (Test-Path -Path $unc) {
            try {
                Get-ChildItem -Path $unc -ErrorAction Stop | Out-Null
                $ok++
                Write-Log ("SUCCESS: " + $unc + " accessible") "Success"

                if ($ApplyPermissions) {
                    if (-not (Test-IsAdminShare -UNC $unc)) {
                        if ($principal) {
                            $shareName = ($unc -split '\\')[3]
                            $remoteFolder = "\\$srv\C$\Atria\Temp"
                            $remoteScriptPath = "C:\Atria\Temp\Update-AtriaFSSSharePermissions.ps1"

                            Write-Log ("Preparing Atria temp folder on " + $srv + "...")
                            Invoke-Command -ComputerName $srv -ScriptBlock {
                                param($folder)
                                if (-not (Test-Path $folder)) { New-Item -ItemType Directory -Force -Path $folder | Out-Null }
                            } -ArgumentList "C:\Atria\Temp" -ErrorAction SilentlyContinue

                            Write-Log ("Copying permission script to " + $remoteFolder + " ...")
                            Copy-Item -Path $PermScript -Destination $remoteFolder -Force

                            Write-Log ("Executing permission update on " + $srv + " for share " + $shareName + "...")
                            try {
                                $remoteOutput = Invoke-Command -ComputerName $srv -ScriptBlock {
                                    param($script,$share,$principal)
                                    & $script -ShareName $share -Principal $principal 2>&1
                                } -ArgumentList $remoteScriptPath,$shareName,$principal -ErrorAction Stop

                                if ($remoteOutput) {
                                    foreach ($line in $remoteOutput) {
                                        if ($null -ne $line) {
                                            $text = $line.ToString().Trim()
                                            if ($text -ne "") { Write-Log ("[" + $srv + "] " + $text) }
                                        }
                                    }
                                }
                                Write-Log ("Permission update completed successfully on " + $srv + " for " + $unc) "Success"
                            }
                            catch {
                                Write-Log ("ERROR applying permissions on " + $srv + ": " + $_.Exception.Message) "Error"
                            }

                            Write-Log ("Cleaning up temp script on " + $srv + "...")
                            Invoke-Command -ComputerName $srv -ScriptBlock {
                                param($path)
                                if (Test-Path $path) { Remove-Item -Path $path -Force }
                            } -ArgumentList $remoteScriptPath -ErrorAction SilentlyContinue
                        }
                        else {
                            Write-Log ("Skipped permission update for " + $unc + " (domain unknown)") "Warning"
                        }
                    }
                    else {
                        Write-Log ("Skipped admin share: " + $unc) "Warning"
                    }
                }
                else {
                    Write-Log ("Permission update skipped for " + $unc + " (dry-run mode)")
                }
            }
            catch {
                $limited++
                Write-Log ("SUCCESS (Limited): " + $unc + " accessible but limited") "Success"
            }
        }
        else {
            $fail++
            Write-Log ("FAIL: " + $unc + " not accessible") "Error"
        }
    }
    catch {
        $err++
        Write-Log ("ERROR testing " + $unc + ": " + $_.Exception.Message) "Error"
    }
}

Write-Host "==============================" -ForegroundColor Cyan
Add-Content -Path $LogPath -Value "=============================="
Write-Log ("Summary -> OK: " + $ok + ", Limited: " + $limited + ", Fail: " + $fail + ", Error: " + $err)
Write-Log ("Log file saved to: " + $LogPath)
Write-Log ("=== Completed share accessibility test for [" + $Service + "] ===") "Success"
